//
//  TodoListEditor.swift
//  Do It
//
//  Created by Jim Dovey on 2/4/20.
//  Copyright © 2020 Jim Dovey. All rights reserved.
//

import SwiftUI

struct TodoListEditor: View {
    // START:NewState
    @Environment(\.presentationMode) private var presentation
    @Environment(\.managedObjectContext) private var objectContext
    @ObservedObject var list: TodoItemList
    // END:NewState

    var iconGradient: LinearGradient {
        var (h, s, b) = list.color.hsb
        b *= 0.7
        
        return LinearGradient(
            gradient: Gradient(colors: [
                list.color.uiColor,
                Color(hue: h, saturation: s, brightness: b)
            ]),
            startPoint: .topLeading,
            endPoint: .bottom)
    }

    var body: some View {
        VStack {
            HStack(alignment: .firstTextBaseline) {
                Button("Cancel") {
                    self.presentation.wrappedValue.dismiss()
                }
                Spacer()
                Text("Name & Appearance")
                    .bold()
                Spacer()
                Button(action: {
                    // START:DoneButtonAction
                    try? self.objectContext.save()
                    self.presentation.wrappedValue.dismiss()
                    // END:DoneButtonAction
                }) {
                    Text("Done")
                        .bold()
                }
            }
            .padding()

            // START:ImageNilHandling
            Image(systemName: list.icon ?? "list.bullet")
                // END:ImageNilHandling
                .font(.system(size: 56, weight: .bold, design: .rounded))
                .aspectRatio(contentMode: .fit)
                .padding(36)
                .foregroundColor(.white)
                .frame(width: 112, height: 112)
                .background(iconGradient)
                .clipShape(Circle())
                .doubleShadow()

            // START:TitleNilHandling
            TextField("List Title", text: Binding($list.name, "New List"))
                // END:TitleNilHandling
                .filledPlatter()
                .font(.system(size: 20, weight: .semibold, design: .rounded))
                .padding()
            
            VStack(spacing: 0) {
                Divider()
                ScrollView {
                    VStack {
                        ColorPicker(selectedColor: $list.color)
                            .padding()
                        // START:IconChooserNilHandling
                        IconChooser(selectedIcon: Binding($list.icon, "list.bullet"))
                            // END:IconChooserNilHandling
                            .padding()
                    }
                }
            }
        }
    }
}

struct TodoListEditor_Previews: PreviewProvider {
    static var previews: some View {
        // START:Preview
        let context = PreviewDataStore.shared.viewContext
        return TodoListEditor(list: TodoItemList.defaultList(in: context))
            .environment(\.managedObjectContext, context)
        // END:Preview
    }
}
